﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using STNCInterpreter;
using STNCAnalyzer;
using STNCFormers;
using System.Runtime.InteropServices;

// Example of an interpreter implementing a drilling cycle CYCLE81 (Sinumerik 840D mill)
namespace sample_dc
{
    // Main class
    public class Interpreter : MarshalByRefObject
    {
        public INCT_Interpreter NCT_GetInterface()
        {
            return new TInterpreter();
        }

        public void NCT_FreeMemory()
        {
            GC.Collect();
            GC.WaitForPendingFinalizers();
        }
    }

    // List of delayed commands
    public enum TDelayedCmd
    {
        dcDoDrillingCycle
    }

    // Interpreter class
    public class TInterpreter : MarshalByRefObject, INCT_Interpreter
    {
        // SprutCAM interfaces
        private INCT_BlockExecArray BEA;
        private INCT_SysState SS;
        private INCT_Equipment EQ;
        private INCT_CLData CLD;
        private INCT_Analyzer Analyzer;

        // Object that implements work with the drilling cycle former
        private TDrillingCycles DC;
        // Saved tool position (Y axis)
        private double CurYPos;

        public TInterpreter()
        {
            DC = new TDrillingCycles();
            CurYPos = 0;
        }

        ~TInterpreter()
        {
            BEA = null;
            SS = null;
            EQ = null;
            CLD = null;
            Analyzer = null;
        }

        // The function should return a SprutCAM string with the interpreter library identifier.
        public string GetIdentifier()
        {
            return "{A1F9AC02-DD70-41E2-8139-A2C443F21F79}";
        }

        // The function should return the SprutCAM interface INCT_RegisterList, 
        // which implements the list of interpreter registers
        public INCT_RegisterList GetRegisters()
        {
            // In this example, registers are not needed
            return null;
        }

        // The method should receive and, if necessary to the interpreter, 
        // save the interfaces transmitted from SprutCAM for subsequent use. 
        // The method will be called several times: according to the number of transmitted interfaces.
        public void SetInterface(object Intf)
        {
            if (Intf is INCT_BlockExecArray) { BEA = Intf as INCT_BlockExecArray; }
            if (Intf is INCT_SysState) { SS = Intf as INCT_SysState; }
            if (Intf is INCT_Equipment) { EQ = Intf as INCT_Equipment; }
            if (Intf is INCT_CLData) { CLD = Intf as INCT_CLData; }
            if (Intf is INCT_Analyzer) { Analyzer = Intf as INCT_Analyzer; }
        }

        // The method informs the interpreter about the completion of the transfer of all 
        // SprutCAM interfaces and the ability to perform initialization of the interpreter.
        public void Initialize()
        {
            Analyzer.BeginUpdate();
            try
            {
                Analyzer.LineBeginWithNumber = false;
                Analyzer.Comments.Add("[", "]");
                Analyzer.Comments.Add("(*", "*)");
                Analyzer.Comments.Add("!", "");
                Analyzer.Comments.Add("%", "");
                Analyzer.Comments.Add(";", "");
                Analyzer.Brackets.Add('(', ')');
            }
            finally
            {
                Analyzer.EndUpdate();
            }

            // Initialization 
            DC.Init(CLD, GetDCAxisIndexes());
        }

        // The method informs the interpreter about the interpretation of the next operation
        public void NextOperation()
        {
        }

        // The method passes the frame of the control program to the interpreter (or a part of the frame, 
        // depending on the current iteration of the frame translation) to attempt to read the lexeme or the 
        // semantic construction from its beginning. In case the interpreter has recognized a token or a structure, 
        // it must return the number of characters from the beginning of the line that the recognized lexeme or 
        // structure constitutes. Otherwise, returns an amount equal to 0.
        public void TransLine(string CurrentLine, out int ProcessedChars, out bool NextLine, out bool BreakLine)
        {
            ProcessedChars = 0;
            NextLine = false;
            BreakLine = false;

            if (IsCycle(CurrentLine, ref ProcessedChars))
            {
                return;
            }
        }

        // The function should perform a register search by address, return the search result and the 
        // index of the found register.
        public bool DefineRegister(string Addr, out int Index)
        {
            Index = -1;
            return false;
        }

        // The method informs the interpreter about the end of the current frame translation 
        // and the beginning of its interpretation.
        public void BeforeInterprete()
        {
        }

        // The method passes to the interpreter the element number from the list of objects 
        // for interpreting the current frame. The interpreter must interpret the transmitted object, 
        // i.e. by object number, determine its type and, if necessary, add a command to form the 
        // element(s) of the tool path.
        public void Interprete(int ExecNum)
        {
            INCT_BlockExecObject BEO;

            // Get block exec object by ExecNum
            BEO = BEA.Get(ExecNum);
            try
            {
                switch (BEO.ExecType)
                {
                    // Is other object
                    case TNCExecObj.exObj:
                        switch ((TDelayedCmd)BEO.ExecParam)
                        {
                            case TDelayedCmd.dcDoDrillingCycle:
                                CmdDoDrillingCycle();
                                break;
                        }
                        break;
                }
            }
            finally
            {
                BEO = null;
            }
        }

        // The method informs the interpreter about the end of the interpretation 
        // of the current frame.
        public void AfterInterprete()
        {
        }

        // The function should return the end of the program to SprutCAM if the end of the program 
        // command was processed when interpreting the text of the G-code. Translation and interpretation 
        // of the G-code is completed.
        public bool EndOfProgram()
        {
            return false;
        }

        // Adding a command to BEA (to the end of the list). For further interpretation.
        public void AddDelayedCmd(TDelayedCmd Cmd)
        {
            BEA.Add(TNCExecObj.exObj, (IntPtr) Cmd);
        }

        // Skipping spaces for the line
        public void SkipSpaces(string ALine, ref int ANumChars)
        {
            while ((ANumChars < ALine.Length) && char.IsWhiteSpace(ALine[ANumChars]))
            {
                ANumChars = ANumChars + 1;
            }
        }

        // Try to get ident from the line
        public bool GetIdent(in string ALine, out string AIdent, ref int ANumChars)
        {
            SkipSpaces(ALine, ref ANumChars);

            string Ident = "";
            int NumChars = ANumChars;
            while ((NumChars < ALine.Length) && char.IsLetter(ALine[NumChars]))
            {
                Ident = Ident + ALine[NumChars];
                NumChars = NumChars + 1;
            }

            if (Ident == "")
            {
                AIdent = "";
                return false;
            }
            else
            {
                AIdent = Ident;
                ANumChars = NumChars;
                return true;
            }
        }

        // Try to get integer from the line
        public bool GetInteger(in string ALine, out int ANum, ref int ANumChars)
        {
            SkipSpaces(ALine, ref ANumChars);

            string Num = "";
            int NumChars = ANumChars;
            while ((NumChars < ALine.Length) && (char.IsNumber(ALine[NumChars]) || ALine[NumChars].Equals("-")))
            {
                Num = Num + ALine[NumChars];
                NumChars = NumChars + 1;
            }

            try
            {
                ANum = Int32.Parse(Num);
                ANumChars = NumChars;
                return true;
            }
            catch
            {
                ANum = 0;
                return false;
            }
        }

        // Try to get sym from line
        public bool GetSym(in string ALine, in int ANum, out string ASym, ref int ANumChars)
        {
            SkipSpaces(ALine, ref ANumChars);

            bool res = ALine.Length > ANumChars + ANum - 1;
            if (res)
            {
                ASym = ALine.Substring(ANumChars, ANum);
                ANumChars = ANumChars + ANum;
                return true;
            }
            else
            {
                ASym = "";
                return false;
            }
        }

        // Try to get double from the line
        public bool GetDouble(in string ALine, out double ANum, ref int ANumChars)
        {
            SkipSpaces(ALine, ref ANumChars);

            string Num = "";
            int NumChars = ANumChars;
            while ((NumChars < ALine.Length) && (char.IsNumber(ALine[NumChars]) || ALine[NumChars].Equals("-") || ALine[NumChars].Equals(".")))
            {
                Num = Num + ALine[NumChars];
                NumChars = NumChars + 1;
            }

            try
            {
                ANum = double.Parse(Num);
                ANumChars = NumChars;
                return true;
            }
            catch
            {
                ANum = 0;
                return false;
            }
        }

        // Attempt to recognize "CYCLE" in line
        public bool IsCycle(string CurrentLine, ref int ProcessedChars)
        {
            string Sym;
            int CycleNum;
            double Num;
            
            string Line = CurrentLine;
            int NumChars = 0;

            bool res = false;

            if (!GetIdent(Line, out Sym, ref NumChars))
            {
                return res;
            }
            if (String.Compare(Sym, "CYCLE", StringComparison.OrdinalIgnoreCase) != 0)
            {
                return res;
            }
            if (!GetInteger(Line, out CycleNum, ref NumChars))
            {
                return res;
            }
            if (!GetSym(Line, 1, out Sym, ref NumChars))
            {
                return res;
            }
            if (Sym != "(")
            {
                return res;
            }

            DC.SetCycleNum(CycleNum);
            try
            {
                try
                {
                    do
                    {
                        bool Dbl = GetDouble(Line, out Num, ref NumChars);
                        if (!Dbl)
                        {
                            Num = 0;
                        };
                        if (!GetSym(Line, 1, out Sym, ref NumChars))
                        {
                            return res;
                        };
                        if ((Sym != ",") && (Sym != ")"))
                        {
                            return res;
                        }
                        DC.SetNextParameter(!Dbl, Num);                   
                    } while (Sym != ")");

                    AddDelayedCmd(TDelayedCmd.dcDoDrillingCycle);
                    ProcessedChars = NumChars;
                    res = true;
                }
                finally
                {
                    if (!res)
                    {
                        DC.Cancel();
                    }
                }

            }
            catch
            {
                ProcessedChars = 0; 
                return res;
            }
            return res;
        }

        // Get axis indexes
        public TNCDCAxisIndexes GetDCAxisIndexes()
        {
            TNCDCAxisIndexes res;
            res.X = EQ.GetAxisIndex("X");
            res.Y = EQ.GetAxisIndex("Y");
            res.Z = EQ.GetAxisIndex("Z");
            return res;
        }

        // Set initial tool position
        public void SetInitialPos()
        {
            CLD.OpenMultiMotion(TNCGotoMode.gtmTooltip);
            CLD.AddMotion(EQ.GetAxisIndex("X"), 0);
            CLD.AddMotion(EQ.GetAxisIndex("Y"), CurYPos);
            CLD.AddMotion(EQ.GetAxisIndex("Z"), DC.GetBaseLevel());
            CLD.CloseMultiMotion();

            CurYPos = CurYPos + 100;
        }

        public void CmdDoDrillingCycle()
        {
            // Artificially set the initial position of the tool
            SetInitialPos();
            
            // Set init level
            DC.SetInitLevel();

            // Formation toolpath
            DC.DoCycle();
        }
    }
}
